/*
   COPYRIGHT NOTICE
   
   This code was originally written, and is owned by Brian Somers.
   It may be copied, altered, given away or sold by anybody who
   feels so inclined, but must at all times contain this copyright
   notice.
   brian@awfulhak.demon.co.uk
   
   */
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "balance.inc"
#	endif
#   endif
#else
#   include "balance.h"
#   include "include/malloc.h"
#   include "include/memory.h"
#endif

static int move_left(_balanced_tree *this);
static int move_right(_balanced_tree *this);
static int move_back(_balanced_tree *this);
static void adjust_factors(_balanced_tree *this);
static _balanced_tree_node *balance_act(_balanced_tree *this);
static int find(_balanced_tree *this,_balanced_tree_key *key);
static short height(const _balanced_tree *this,
					const _balanced_tree_node *node);
static short validate(_balanced_tree *this,_balanced_tree_node *node);
static _balanced_tree_node *newnode(const void *v,int len,
										  const void *data,int do_malloc);
static void Exchange(_balanced_tree_node **n1,_balanced_tree_node **n2);
static void delnode(_balanced_tree_node *this);

static int move_left(_balanced_tree *this){
	if(this->treeerror) return 0;
	if(this->position->l){
		this->position = this->position->l;
		this->path[this->plen++] = this->position;
		if(this->plen != this->max_plen) return 1;
		this->max_plen += 10;
		this->path = (_balanced_tree_node **)
			mem_realloc(this->path,this->max_plen * sizeof(_balanced_tree_node *));
		if(this->path) return 1;
		this->treeerror = 1;
	}
	return 0;
}

static int move_right(_balanced_tree *this){
	if(this->treeerror) return 0;
	if(this->position->r){
		this->position = this->position->r;
		this->path[this->plen++] = this->position;
		if(this->plen != this->max_plen) return 1;
		this->max_plen += 10;
		this->path = (_balanced_tree_node **)
			mem_realloc(this->path,this->max_plen * sizeof(_balanced_tree_node *));
		if(this->path) return 1;
		this->treeerror = 2;
	}
	return 0;
}

static int move_back(_balanced_tree *this){
	if(this->treeerror || this->plen == 1) return 0;
	this->position = this->path[this->plen-2];
	this->plen--;
	return 1;
}

typedef unsigned char *uchar;

int baltree_keycmp(const _balanced_tree_key *k1,const _balanced_tree_key *k2,
				   int fast){
	size_t len, f;
	int ret = 0;

	if(k1 == k2) return 0;
	if(fast){
		if( k1->length < k2->length ) return -1;
		else if( k1->length > k2->length ) return 1;
		else for( f = k1->length - 1; f != -1; f-- ){
			if(*((char *)k1->data+f) < *((char *)k2->data+f)) return -1;
			else if(*((char *)k1->data+f) > *((char *)k2->data+f)) return 1;
		}
		return ret;
	}

	len = k1->length < k2->length ? k1->length : k2->length;

	for( f = 0; f < len; f++ )
		if(*((char *)k1->data + f) != *((char *)k2->data + f)){
			if(*((uchar)k1->data + f) < *((uchar)k2->data + f)) ret = -1;
			else ret = 1;
			break;
		}
	if(ret || k1->length == k2->length) return ret;
	return k1->length == len ? -1 : 1;
}

static void adjust_factors(_balanced_tree *this){
	_balanced_tree_node *bal = this->bal_at ? this->bal_at : this->adjust_at;
	if(this->treeerror || !bal) return;

	while(bal != this->position){
		switch(baltree_keycmp(&this->position->key,&bal->key,
							  this->flag & BALTREE_FAST_KEYS)){
		case -1:
			bal->factor--;
			bal = bal->l;
			break;
		case 1:
			bal->factor++;
			bal = bal->r;
		}
		if(!bal){ this->treeerror = 3; break; }
	}
}

static _balanced_tree_node *balance_act(_balanced_tree *this){
	_balanced_tree_node *bal = this->bal_at, *parent;
	if(this->treeerror || !bal) return 0;

	if(bal->factor == -2){
		if(bal->l->factor < 0){	/* single rotation */
			parent = bal->l;
			bal->l = parent->r;
			parent->r = bal;
			parent->factor = parent->r->factor = 0;
		}else if(bal->l->factor > 0){ /* double rotation */
			parent = bal->l->r;
			bal->l->r = parent->l;
			parent->l = bal->l;
			bal->l = parent->r;
			parent->r = bal;
			if(parent->factor > 0){
				parent->r->factor = 0;
				parent->l->factor = -1;
			}else if(parent->factor < 0){
				parent->r->factor = 1;
				parent->l->factor = 0;
			}else
				parent->r->factor = parent->l->factor = 0;
			parent->factor = 0;
		}else{					/* special deletion case */
			parent = bal->l;
			bal->l = parent->r;
			parent->r = bal;
			parent->r->factor = -1;
			parent->factor = 1;
		}
	}else if(bal->factor == 2){
		if(bal->r->factor > 0){	/* single rotation */
			parent = bal->r;
			bal->r = parent->l;
			parent->l = bal;
			parent->factor = parent->l->factor = 0;
		}else if(bal->r->factor < 0){ /* double rotation */
			parent = bal->r->l;
			bal->r->l = parent->r;
			parent->r = bal->r;
			bal->r = parent->l;
			parent->l = bal;
			if(parent->factor > 0){
				parent->r->factor = 0;
				parent->l->factor = -1;
			}else if(parent->factor < 0){
				parent->r->factor = 1;
				parent->l->factor = 0;
			}else
				parent->r->factor = parent->l->factor = 0;
			parent->factor = 0;
		}else{					/* special deletion case */
			parent = bal->r;
			bal->r = parent->l;
			parent->l = bal;
			parent->l->factor = 1;
			parent->factor = -1;
		}
	}else{
		this->treeerror = 4;
		return 0;
	}

	if(!this->bal_parent)
		this->top_node = parent;
	else if(this->bal_parent->l == this->bal_at)
		this->bal_parent->l = parent;
	else if(this->bal_parent->r == this->bal_at)
		this->bal_parent->r = parent;
	else this->treeerror = 5;

	return parent;
}

static int find(_balanced_tree *this,_balanced_tree_key *key){
	_balanced_tree_node *last = 0;
	if(this->treeerror) return -1;
	this->position = this->adjust_at = this->top_node;
	if(!this->top_node) return -2;
	this->bal_at = this->bal_parent = 0;
	this->plen = 1;
	*this->path = this->top_node;
	
	while(1){
		switch(baltree_keycmp(key,&this->position->key,
							  this->flag & BALTREE_FAST_KEYS)){
		case -1:
			if(this->position->factor == -1){
				this->bal_at = this->position;
				this->bal_parent = last;
			}else if(this->position->factor == 1){
				this->bal_at = this->bal_parent = 0;
				this->adjust_at = this->position;
			}
			last = this->position;
			if(!move_left(this)) return -1;
			if(this->treeerror) return -2;
			break;
		case 1:
			if(this->position->factor == 1){
				this->bal_at = this->position;
				this->bal_parent = last;
			}else if(this->position->factor == -1){
				this->bal_at = this->bal_parent = 0;
				this->adjust_at = this->position;
			}
			last = this->position;
			if(!move_right(this)) return 1;
			break;
		default:
			return 0;
		}
	}
}

static short height(const _balanced_tree *this,
					const _balanced_tree_node *node){
	short level_l, level_r;
	if(this->treeerror || !node) return 0;
	level_l = height(this,node->l);
	level_r = height(this,node->r);
	return level_l > level_r ? level_l + 1 : level_r + 1;
}

static short validate(_balanced_tree *this,_balanced_tree_node *node){
	short level_l, level_r;
	if(this->treeerror) return -1;
	if(!node) return 0;
	if(level_l = validate(this,node->l), level_l == -1) return -1;
	if(level_r = validate(this,node->r), level_r == -1) return -1;
	if(level_r - level_l != node->factor){
		find(this,&node->key);
		return -1;
	}
	return level_l > level_r ? level_l + 1 : level_r + 1;	 
}

static void delnode(_balanced_tree_node *this){
	if(this->l) delnode(this->l);
	if(this->r) delnode(this->r);
	mem_free(this);
}

void baltree_construct(_balanced_tree *this,int flag){
	this->bal_at = this->bal_parent = this->position = this->top_node = 0;
	this->treeerror = this->plen = 0;
	this->max_plen = 10;		/* This will expand if needed */
	this->path = (_balanced_tree_node **)
		mem_malloc(this->max_plen * sizeof(_balanced_tree_node *));
	this->flag = flag;
	this->pos = 0;
	if(!this->path) this->treeerror = 14;
}

void baltree_destroy(_balanced_tree *this){
	if(this->treeerror) return;
	if(this->top_node)
		delnode(this->top_node);
	if(this->path) mem_free(this->path);
}

short baltree_error(const _balanced_tree *this){
	return this->treeerror;
}

short baltree_height(const _balanced_tree *this){
	return height(this,this->top_node);
}

int baltree_validate(_balanced_tree *this){
	return validate(this,this->top_node) == -1 ? 0 : 1;
}

const void *baltree_get(_balanced_tree *this,const void *v,int len){
	_balanced_tree_key key;
	key.length = len;
	key.data = (void *)v;	/* It WON'T be changed by baltree ! */
	if(this->treeerror) return 0;
	if(this->top_node && !find(this,&key)){
		this->pos = 0;
		return this->position->data;
	}
	return 0;
}

static _balanced_tree_node *newnode(const void *v, int len,
									const void *data, int do_malloc){
	_balanced_tree_node *node = (_balanced_tree_node *)
		mem_malloc(sizeof(_balanced_tree_node) + (do_malloc ? len : 0));
	if(node){
		node->l = node->r = 0;
		if(do_malloc){
			node->key.data = (void *)((char *)node +
									  sizeof(_balanced_tree_node));
			memcpy(node->key.data,v,len);
		}else node->key.data = (void *)v;	/* I WON'T change it */
		node->key.length = len;
		node->data = data;
		node->factor = 0;
	}
	return node;
}

static void Exchange(_balanced_tree_node **n1,_balanced_tree_node **n2){
	_balanced_tree_node *n = *n1;
	_balanced_tree_node *l = (*n1)->l;
	_balanced_tree_node *r = (*n1)->r;
	unsigned char factor = (*n1)->factor;
	(*n1)->l = (*n2)->l;
	(*n1)->r = (*n2)->r;
	(*n1)->factor = (*n2)->factor;
	(*n2)->l = l;
	(*n2)->r = r;
	(*n2)->factor = factor;
	*n1 = *n2;
	*n2 = n;
}

int baltree_insert(_balanced_tree *this,const void *v,int len,
				   const void *data){
	_balanced_tree_key key;
	if(this->treeerror || !len) return 0;
	key.length = len;
	key.data = (void *)v;	/* I WON'T change it */
	
	if(!this->top_node){
		this->pos = 0;
		*this->path = this->position = this->top_node =
			newnode(v,len,data,this->flag & BALTREE_MALLOC_KEYS);
		return this->plen = 1;
	}
	
	switch(find(this,&key)){
	case 0:
		return 0;
	case -1:
		if(this->treeerror) return 0;
		this->position->l = newnode(v,len,data,this->flag
									& BALTREE_MALLOC_KEYS);
		if(!move_left(this) || this->treeerror) return 0;
		break;
	case 1:
		if(this->treeerror) return 0;
		this->position->r = newnode(v,len,data,this->flag
									& BALTREE_MALLOC_KEYS);
		if(!move_right(this) || this->treeerror) return 0;
		break;
	}
	
	if(!this->treeerror) adjust_factors(this);
	if(this->bal_at){
		if(!this->treeerror) balance_act(this);
		if(!this->treeerror) find(this,&key);
	}

	if(this->treeerror) return 0;
	this->pos = 0;
	return 1;
}

const void *baltree_replace(_balanced_tree *this,const void *v,
							int len,const void *data){
	const void *ret;
	_balanced_tree_key key;
	key.length = len;
	key.data = (void *)v;	/* I WON'T change it */
	if(this->treeerror || find(this,&key)) return 0;
	ret = this->position->data;
	this->position->data = data;
	return ret;
}

const void *baltree_put(_balanced_tree *this,const void *v,
						int len,const void *data){
	const void *ret;
	if(this->treeerror || !len) return 0;
	if(baltree_insert(this,v,len,data)) return 0;
	if(this->treeerror) return 0;
	if(ret = baltree_replace(this,v,len,data), ret) return ret;
	if(!this->treeerror) this->treeerror = 6;
	return 0;
}

const void *baltree_del(_balanced_tree *this,const void *v,int len){
	_balanced_tree_node **old_pos = 0;
	_balanced_tree_key key;
	int special_addition = -1;
	const void *ret;
	_balanced_tree_node *sub;
	key.length = len;
	key.data = (void *)v;	/* I WON'T change it */
	if(this->treeerror || find(this,&key)) return 0;
	
	if(this->position->l && this->position->r){
		int old_len = this->plen - 1;
		old_pos = this->plen == 1 ? &this->top_node :
			this->path[this->plen-2]->l == this->position ?
				&this->path[this->plen-2]->l : &this->path[this->plen-2]->r;
		this->pos = 0;
		if(this->position->factor == -1){
			special_addition = 1;
			if(!baltree_prev(this)){
				if(!this->treeerror) this->treeerror = 7;
				return 0;
			}
		}else if(!baltree_next(this)){
			if(!this->treeerror) this->treeerror = 8;
			return 0;
		}
		if(*old_pos == this->position){
			if(!this->treeerror) this->treeerror = 9;
			return 0;
		}
		/* swap *old_pos and this->position, exchanging l, r and factor */
		Exchange(old_pos,&this->position);
		/* update the path */
		this->path[ old_len ] = *old_pos;
		/* our return pointer is now in the current node */
		ret = this->position->data;
	}
	sub = this->position->l ? this->position->l : this->position->r;
	this->position->l = this->position->r = 0;
	
	if(this->plen == 1){
		ret = this->position->data;
		delnode(this->position);
		this->top_node = sub;
		baltree_setbefore(this,v,len);
		return ret;
	}
	
	if(this->path[this->plen-2]->l == this->position)
		this->path[this->plen-2]->l = sub;
	else if(this->path[this->plen-2]->r == this->position)
		this->path[this->plen-2]->r = sub;
	else{ this->treeerror = 10; return 0; }
	
	ret = this->position->data;
	delnode(this->position);
	
	/*
	   Now, starting with the parent of the dibbled node, adjust the balance
	   factor by +- 1 depending on whether the deleted node is greater or
	   less than the current one.  Do for all nodes until either the top is
	   reached, or a nodes balance factor becomes 1.
	   NOTE: ( 4/1/91 ) if the current node has been swapped, it will be
	   temporarily out of order, hence the int "special_addition".
	   */

	while(--this->plen){
		this->position = this->path[this->plen-1];
		this->position->factor += (old_pos && *old_pos == this->position) ?
			special_addition : baltree_keycmp(&this->position->key,&key,
											  this->flag & BALTREE_FAST_KEYS);
		switch(this->position->factor){
		case -2:
		case 2:
			this->bal_at = this->position;
			if(this->plen == 1) this->bal_parent = 0;
			else this->bal_parent = this->path[this->plen-2];
			if(this->position = balance_act(this), !this->position)
				this->treeerror = 12;
			if(this->treeerror) return 0;
			if(!this->position->factor) break;
		case -1:
		case 1:
			this->plen = 1;
		}
	}
    baltree_setbefore(this,v,len);
    return ret;
}

void baltree_first(_balanced_tree *this){
    if(this->treeerror) return;
    this->position = *this->path = this->top_node;
    if(this->position){
		this->plen = 1;
		while(move_left(this)) if(this->treeerror) return;
    }else this->plen = 0;
    this->pos = -1;
}

void baltree_last(_balanced_tree *this){
    if(this->treeerror) return;
    this->position = *this->path = this->top_node;
    if(this->position){
		this->plen = 1;
		while(move_right(this)) if(this->treeerror) return;
    }else this->plen = 0;
    this->pos = 1;
}

const void *baltree_next(_balanced_tree *this){
    _balanced_tree_node *last, *old_pos;
    int old_plen;
    if(this->treeerror || !this->top_node) return 0;
    if(this->pos == -1){		/* Don't move forward */
		if(!this->position) return 0;
		this->pos = 0;
    }else{
		this->pos = 0;
		if(move_right(this)){
			while(move_left(this)) if(this->treeerror) return 0;
		}else{
			if(this->treeerror) return 0;
			old_plen = this->plen;
			old_pos = this->position;
			while(1){
				last = this->position;
				if(!move_back(this)){
					this->plen = old_plen;
					this->position = old_pos;
					this->pos = 1;
					return 0;
				}
				if(this->position->l == last) break;
			}
		}
    }
    return this->position->data;
}

const void *baltree_prev(_balanced_tree *this){
    _balanced_tree_node *last, *old_pos;
    int old_plen;
    if(this->treeerror || !this->top_node) return 0;
    if(this->pos == 1){			/* Don't move back */
		if(!this->position) return 0;
		this->pos = 0;
    }else{
		this->pos = 0;
		if(move_left(this)){
			while(move_right(this)) if(this->treeerror) return 0;
		}else{
			if(this->treeerror) return 0;
			old_plen = this->plen;
			old_pos = this->position;
			while(1){
				last = this->position;
				if(!move_back(this)){
					this->plen = old_plen;
					this->position = old_pos;
					this->pos = -1;
					return 0;
				}
				if(this->position->r == last) break;
			}
		}
    }
    return this->position->data;
}

void baltree_setbefore(_balanced_tree *this,const void *v,int len){
    _balanced_tree_key key;
    if(this->treeerror) return;
    key.length = len;
    key.data = (void *)v;	/* I WON'T change it */
    switch(find(this,&key)){
	case  1:
		this->pos = 1;
	case -2:
		break;
	default:
		this->pos = -1;
    }
}

void baltree_setafter(_balanced_tree *this,const void *v,int len){
    _balanced_tree_key key;
    if(this->treeerror) return;
    key.length = len;
    key.data = (void *)v;	/* I WON'T change it */
    switch(find(this,&key)){
	case -1:
		this->pos = -1;
	case -2:
		break;
	default:
		this->pos = 1;
    }
}

const _balanced_tree_key *baltree_topnodekey(const _balanced_tree *this){
    if(!this->top_node || this->treeerror) return 0;
    return &this->top_node->key;
}

const _balanced_tree_key *baltree_key(const _balanced_tree *this){
    if(this->treeerror || !this->plen) return 0;
    return &this->path[this->plen-1]->key;
}
