//
//                      COPYRIGHT NOTICE
//
//      This code was originally written, and is owned by Brian Somers.
//      It may be copied, altered, given away or sold by anybody who
//      feels so inclined, but must at all times contain this copyright
//      notice.
//      brian@awfulhak.demon.co.uk
//
//
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "signals.inc"
#	endif
#   endif
#else
#   include "signals.h"
#   include "include/stdarg.h"
#   include "include/stdlib.h"
#   include "include/malloc.h"
#   include "include/stdio.h"
#   include "include/unistd.h"
#endif

#ifdef MSDOS
#   if !defined(__ZTC__) && !defined(__WATCOMC__)
	void abort(void){ exit(6); }
#   endif
#endif

typedef Sig_Handler_Return (*handlerptr)(int);

#if defined(DEBUG) && !defined(MSDOS)

handlerptr mysignal(int sig,handlerptr fn){
    fprintf( stderr, "signals: signal( %d, %p ); // ", sig, fn );
    if(fn == signals::user_stack_handler)
	fputs( "signals::user_stack_handler\n", stderr );
    else if(fn == signals::user_fn_handler)
	fputs( "signals::user_fn_handler\n", stderr );
    else if(fn == signals::jmp_handler)
	fputs( "signals::jmp_handler\n", stderr );
    else if(fn == signals::save_handler)
	fputs( "signals::save_handler\n", stderr );
    else if(fn == SIG_IGN)
	fputs( "SIG_IGN\n", stderr );
    else if(fn == SIG_DFL)
	fputs( "SIG_DFL\n", stderr );
    else
	fputs( "unknown function\n", stderr );
    return signal(sig,fn);
}

#define signal mysignal
#endif	// DEBUG && !MSDOS

signals::signals(int i,...){
    va_list ap;
    int sig = i;

    this->maxsig = 0;
    va_start(ap,i);
    do
	if(this->maxsig < sig) this->maxsig = sig;
    while(sig = va_arg(ap,int), sig);

    this->previous_handler = mem_new handlerptr[this->maxsig];
    this->caused = mem_new int[this->maxsig];
    this->handling = mem_new int[this->maxsig];
    for(sig = 0; sig < this->maxsig; sig++) this->handling[sig] = 0;

    sig = i;
    va_start(ap,i);
    do{
	this->previous_handler[sig-1] = signal(sig,SIG_IGN);
	signal(sig,this->previous_handler[sig-1]);
	this->handling[sig-1] = 1;
    }while(sig = va_arg(ap,int), sig);

    this->user_fn = 0;
    this->user_jmpbuf = 0;
    this->user_save = 0;
    this->user_stack = 0;
    this->isactive = 0;

    if(!signals::declared){
	signals::maxdeclared = 10;
	signals::declared = (signals **)mem_malloc(signals::maxdeclared
					       * sizeof(signals *));
    }else if(signals::ndeclared == signals::maxdeclared){
	signals::maxdeclared += 10;
	signals::declared = (signals **)mem_realloc(signals::declared,
						signals::maxdeclared
						* sizeof(signals *));
    }
    signals::declared[signals::ndeclared++] = this;
}

signals::~signals(){
    this->restore();
    mem_delete this->previous_handler;
    mem_delete this->caused;
    mem_delete this->handling;
    if(signals::declared[--signals::ndeclared] != this){
	fputs( "Un-nested signals object.... Abort\n", stderr );
	abort();
    }
    if(!signals::ndeclared){
	mem_free(signals::declared);
	signals::maxdeclared = 0;
	signals::declared = 0;
    }
}

Sig_Handler_Return signals::save_handler(int sig){
    signal(sig,SIG_IGN);
    signals *object = signals::active(sig,SAVE_HANDL);
    if(!object){
	fprintf( stderr, "Sig %d: Cannot find active SAVE handler\n", sig );
	abort();
    }
    object->caused[sig-1]++;
    signal(sig,signals::save_handler);
}

Sig_Handler_Return signals::jmp_handler(int sig){
    signal(sig,SIG_IGN);
    signals *object = signals::active(sig,JMP_HANDL);
    if(!object){
	fprintf( stderr, "Sig %d: Cannot find active JMP handler\n", sig );
	abort();
    }
    signal(sig,signals::jmp_handler);
    longjmp(*object->user_jmpbuf,sig);
}

Sig_Handler_Return signals::user_fn_handler(int sig){
    signal(sig,SIG_IGN);
    signals *object = signals::active(sig,USER_FN_HANDL);
    if(!object){
	fprintf( stderr, "Sig %d: Cannot find active USER_FN handler\n", sig );
	abort();
    }
    (*object->user_fn)(sig);
    signal(sig,signals::user_fn_handler);
}

Sig_Handler_Return signals::user_stack_handler(int sig){
    int recursing = signal(sig,SIG_IGN) == SIG_IGN;
    signals *object = signals::active(sig,USER_STACK_HANDL);
    if(!object){
	fprintf( stderr, "Sig %d: Cannot find active "
		"USER_STACK handler\n", sig );
	abort();
    }
    (*object->user_stack)(sig);
    object->isactive = 0;
    object->call_oldhandler(sig);
    object->isactive = 1;
    if(!recursing) signal(sig,signals::user_stack_handler);
}

signals *signals::active(int sig,active_handler type){
    for(int f = 0; f < signals::ndeclared; f++)
	if(signals::declared[f]->isactive
	   && signals::declared[f]->handling[sig-1])
	    switch(type){
	    case SAVE_HANDL:
		if(signals::declared[f]->user_save)
		    return signals::declared[f];
		break;
	    case JMP_HANDL:
		if(signals::declared[f]->user_jmpbuf)
		    return signals::declared[f];
		break;
	    case USER_FN_HANDL:
		if(signals::declared[f]->user_fn)
		    return signals::declared[f];
		break;
	    case USER_STACK_HANDL:
		if(signals::declared[f]->user_stack)
		    return signals::declared[f];
		break;
	    }
    return 0;
}

void signals::uncause(){
    for(int sig = 0; sig < this->maxsig; sig++) this->caused[sig] = 0;
}

void signals::call_oldhandler(int sig){
    handlerptr fn = this->previous_handler[sig-1];
    handlerptr current = signal(sig,SIG_IGN);
    if(fn == SIG_DFL){
	signal(sig,fn);
#ifdef MSDOS
	raise(sig);
#else
	kill(getpid(),sig);
#endif
	signal(sig,current);
    }else if(fn != SIG_IGN) (*fn)(sig);
}

void signals::stack(handlerptr fn){
    if(this->isactive) this->restore();
    this->uncause();
    for(int sig = 0; sig < this->maxsig; sig++)
	if(this->handling[sig]) signal(sig+1,signals::user_stack_handler);
    this->user_stack = fn;
    this->isactive = 1;
}

void signals::save(handlerptr fn){
    if(this->isactive) this->restore();
    this->uncause();
    for(int sig = 0; sig < this->maxsig; sig++)
	if(this->handling[sig]) signal(sig+1,signals::user_fn_handler);
    this->user_fn = fn;
    this->isactive = 1;
}

void signals::save(jmp_buf *j){
    if(this->isactive) this->restore();
    this->uncause();
    for(int sig = 0; sig < this->maxsig; sig++)
	if(this->handling[sig]) signal(sig+1,signals::jmp_handler);
    this->user_jmpbuf = j;
    this->isactive = 1;
}

void signals::save(){
    if(this->isactive) this->restore();
    this->uncause();
    for(int sig = 0; sig < this->maxsig; sig++)
	if(this->handling[sig]) signal(sig+1,signals::save_handler);
    this->user_save = 1;
    this->isactive = 1;
}

void signals::restore(){
    if(!this->isactive) return;
    this->isactive = 0;
    this->user_fn = 0;
    this->user_stack = 0;
    this->user_jmpbuf = 0;
    this->user_save = 0;
    for(int sig = 0; sig < this->maxsig; sig++)
	if(this->handling[sig]) signal(sig+1,this->previous_handler[sig]);
}

int signals::cause(){
    int ret = 0;
    if(this->isactive) return ret;
    for(int sig = 0; sig < this->maxsig; sig++)
	if(this->handling[sig] && this->caused[sig]){
	    ret++;
	    this->call_oldhandler(sig+1);
	}
    return ret;
}

signals **signals::declared = 0;
int signals::ndeclared = 0;
int signals::maxdeclared = 0;
