/*
			COPYRIGHT NOTICE

	This code was originally written, and is owned by Brian Somers.
	It may be copied, altered, given away or sold by anybody who
	feels so inclined, but must at all times contain this copyright
	notice.
        brian@awfulhak.demon.co.uk

*/
#ifdef MK
#   if __GNUC__ < 2
#		if !defined(__ZTC__) && !defined(__WATCOMC__)
#		    include "mem.inc"
#		endif
#   endif
#else
#   include "mem.h"
#   include "assert.h"
#   include "dos.h"
#   include "string.h"
#   include "stdio.h"
#   include "io.h"
#   include "stdarg.h"
#endif

#ifdef MEM_DEBUG

#undef assert
#define assert(x) if((x)) ; else fprintf( stderr, "Assertion failure (%s)\n", #x )

static int mem_inited = 0;

static int mem_behavior = MEM_ABORT;
static int (*fp)(void) = NULL;  /* out-of-memory handler */
static int mem_count = 0;		/* # of allocs that haven't been free'd	*/

#ifdef MSDOS
#	define ferr	stdout
#else
#	define ferr	stderr
#endif
	
	
void MemSetBehaviour(enum MemBehaviour flag,...)
{
	va_list ap;
    typedef int (*fp_t)(void);

    mem_behavior = flag;
    va_start(ap,flag);
    fp = (mem_behavior == MEM_CALLFP) ? va_arg(ap,fp_t) : 0;
    va_end(ap);
}


char *mem_strdup_debug(const char *s,const char *file,int line)
{
	char *p;

	p = s ? (char *) mem_malloc_debug((unsigned) strlen(s) + 1,file,line)
		: NULL;
	return p ? strcpy(p,s) : p;
}


int __mem_line;
char *__mem_file;

/************* C++ Implementation ***************/

static void mem_init(void);
static void mem_term(void);

#ifdef __cplusplus
extern "C++"
{
	/* Cause initialization and termination functions to be called	*/
	static struct cMemDebug
	{
    	cMemDebug(){ mem_init(); }
    	~cMemDebug(){ mem_term(); }
	} dummy;

	void (*_new_handler)(void);

	/*****************************
	 * Replacement for the standard C++ library operator new().
	 */

#undef new
	void *operator new(size_t size)
	{
		void *p;

    	while (1)
    	{
			if( !size )
				size++;
			if( !mem_inited )
				fprintf( stderr, "new called outside mem init !\n" );
			p = mem_malloc_debug(size,__mem_file,__mem_line);
			if( p || !_new_handler )
				break;
			(*_new_handler)();
    	}
    	return p;
	}

	/***********************
	 * Replacement for the standard C++ library operator delete().
	 */

#undef delete
	void operator delete(void *p)
	{
		if( !mem_inited )
			fprintf( stderr, "delete called outside mem init !\n" );
		mem_free_debug(p,__mem_file,__mem_line);
	}

}
#endif


static long mem_maxalloc = 0;	/* max # of bytes allocated		*/
static long mem_numalloc = 0;	/* current # of bytes allocated		*/

#define BEFOREVAL 0x4F464542	/* value to detect underrun	*/
#define AFTERVAL 0x45544641	/* value to detect overrun	*/
const long afterval = AFTERVAL;

/* The following should be selected to give maximum probability that	*/
/* pointers loaded with these values will cause an obvious crash. On	*/
/* Unix machines, a large value will cause a segment fault.		*/
/* MALLOCVAL is the value to set malloc'd data to.			*/

#ifdef MSDOS
#define BADVAL		0xFF
#define MALLOCVAL	0xEE
#else
#define BADVAL		0x7A
#define MALLOCVAL	0xEE
#endif

/* Create a list of all alloc'ed pointers, retaining info about where	*/
/* each alloc came from. This is a real memory and speed hog, but who	*/
/* cares when you've got obscure pointer bugs.				*/

struct mem_debug;
struct mh				/* part of mem_debug struct	*/
{
	struct mem_debug *Mnext;	/* next in list			*/
	struct mem_debug *Mprev;	/* previous value in list	*/
	const char *Mfile;			/* filename of where allocated		*/
	int Mline;					/* line number of where allocated	*/
	unsigned Mnbytes;			/* size of the allocation		*/
	long Mbeforeval;			/* detect underrun of data		*/
};

struct mem_debug
{
	struct mh m;
	char data[1];				/* the data actually allocated		*/
};

static struct FirstInList
{
	struct mem_debug Item;
	long AfterVal;
} firstInList =
{
	/* struct mem_debug */
	{
		/* struct mh */
		{
			(struct mem_debug *)0,
			(struct mem_debug *)0,
			__FILE__, __LINE__,
			1,
			BEFOREVAL
		},
		""
	},
	AFTERVAL
};

#define Offset( Type, Item ) ((int)((( Type * )0)->Item))

#define mem_ptrtodl(p) ((struct mem_debug *)((char *)p - Offset( struct mem_debug, data )))
#define mem_dltoptr(dl)	((void *)(dl)->data)


/****************************
 * Print out struct mem_debug.
 */

static void mem_printdl(struct mem_debug *dl)
{
	fprintf( ferr, "allocated from file %s line %d nbytes %d ptr %p\n",
			dl->m.Mfile, dl->m.Mline, dl->m.Mnbytes, mem_dltoptr(dl) );
}

/****************************
 * Print out file and line number.
 */

static void FreeError( const char *fil, int lin )
{
	fprintf( ferr, "freed from file \"%s\" line %d\n", fil, lin );
	fflush( ferr );
}


/*************************
 * This is called when we're out of memory.
 * Returns:
 *	1:	try again to allocate the memory
 *	0:	give up and return NULL
 */

static int mem_exception()
{
	int behavior;

    behavior = mem_behavior;
	switch (behavior)
	{
		case MEM_ABORT:
		fputs( "Fatal error: out of memory\n", ferr );
		fprintf( ferr, "%ld (really %ld) bytes currently allocated\n",
				mem_numalloc, mem_numalloc + mem_count
				* ( sizeof( struct mem_debug ) + sizeof( afterval ) ) );
		_exit(EXIT_FAILURE);

		case MEM_CALLFP:
		if( !fp ) _exit(EXIT_FAILURE);
		return (*fp)();

		case MEM_RETNULL:
		return 0;
	}
	fputs( "Bad behavior type\n", stderr );
	_exit(1);
	return 0;	/* some compilers complain 'cos exit isn't volatile ! */
}


void *mem_malloc_debug(unsigned n,const char *fil,int lin)
{
    void *p;
    if( p = mem_calloc_debug( n, fil, lin ), p )
		memset( p, MALLOCVAL, n );
    return p;
}

static void mem_paranoid_check( const char *file, int line );

void *mem_calloc_debug(unsigned n,const char *fil,int lin)
{
    struct mem_debug *dl;

    do
	{
#ifdef MEM_PARANOID
		mem_paranoid_check( fil, lin );
#endif
		dl = (struct mem_debug *)
			calloc( sizeof( *dl ) + n + sizeof( afterval ),1);
#ifdef MEM_SHOWCALLS
		fprintf( stderr, "%s: %d: %p = calloc( %u )\n",
				fil, lin, dl ? mem_dltoptr(dl) : (char *)0, n );
#endif
		if( !dl )
			fprintf( stderr, "%s: %d: Cannot allocate %ld bytes !\n",
					fil, lin, (long)sizeof(*dl) + n + sizeof(afterval) );
	}
    while (dl == NULL && mem_exception());
    if( !dl )
		return NULL;
    dl->m.Mfile = fil;
    dl->m.Mline = lin;
    dl->m.Mnbytes = n;
    dl->m.Mbeforeval = BEFOREVAL;
    memcpy(&(dl->data[n]),&afterval,sizeof(afterval));

    /* Add dl to start of allocation list	*/
	dl->m.Mnext = firstInList.Item.m.Mnext;
	dl->m.Mprev = &firstInList.Item;
	firstInList.Item.m.Mnext = dl;

	if( dl->m.Mnext )
		dl->m.Mnext->m.Mprev = dl;

    mem_count++;
    mem_numalloc += n;
    if( mem_numalloc > mem_maxalloc )
		mem_maxalloc = mem_numalloc;
    return mem_dltoptr(dl);
}

void mem_free_debug(void *ptr,const char *fil,int lin)
{
	struct mem_debug *dl;

#ifdef MEM_SHOWCALLS
    fprintf( stderr, "%s: %d: free( %p ) - %u bytes\n",
            fil, lin, ptr, ptr ? mem_ptrtodl( ptr )->m.Mnbytes : (unsigned)0 );
#endif
	if( !ptr )
		return;
	if( mem_count <= 0 )
	{
		fputs( "More frees than allocs - ", ferr );
		FreeError( fil, lin );
		_exit(1);
	}
	dl = mem_ptrtodl(ptr);
	if( dl->m.Mbeforeval != BEFOREVAL )
	{
		struct mem_debug *Item;
		for( Item = firstInList.Item.m.Mnext; Item; Item = Item->m.Mnext )
			if( dl == Item ) break;
		if( Item ){
			fprintf( ferr, "Pointer %p underrun\n", ptr );
			mem_printdl(dl);
		}else fprintf( ferr, "Pointer %p not allocated\n", ptr );
		FreeError( fil, lin );
		_exit(1);
	}
	if( memcmp( &dl->data[dl->m.Mnbytes], &afterval, sizeof( afterval ) ) )
	{
		fprintf( ferr, "Pointer %p overrun\n", ptr );
		mem_printdl(dl);
		FreeError( fil, lin );
		_exit(1);
	}
	mem_numalloc -= dl->m.Mnbytes;
	if( mem_numalloc < 0 )
	{
		fprintf(ferr,"error: mem_numalloc = %ld, dl->nbytes = %d\n",
				mem_numalloc,dl->m.Mnbytes);
		mem_printdl(dl);
		FreeError( fil, lin );
		_exit(1);
	}

	/* Remove dl from linked list	*/
	if( dl->m.Mprev )
		dl->m.Mprev->m.Mnext = dl->m.Mnext;
	if( dl->m.Mnext )
		dl->m.Mnext->m.Mprev = dl->m.Mprev;

	/* Stomp on the freed storage to help detect references	*/
	/* after the storage was freed.				*/
	memset((void *) dl,BADVAL,sizeof(*dl) + dl->m.Mnbytes);
	mem_count--;

	free((void *) dl);
}

void *mem_realloc_debug(void *oldp,unsigned n,const char *fil,int lin)
{
	void *p;
    struct mem_debug *dl;

    if( !n )
    {
		mem_free_debug(oldp,fil,lin);
		p = NULL;
    }
    else if( !oldp )
		p = mem_malloc_debug(n,fil,lin);
    else
    {
		p = mem_malloc_debug(n,fil,lin);
		if( p )
		{
			dl = mem_ptrtodl(oldp);
			if( dl->m.Mnbytes < n )
				n = dl->m.Mnbytes;
			memcpy(p,oldp,n);
			mem_free_debug(oldp,fil,lin);
		}
    }
    return p;
}

/***************************/

static int mem_allocated_ptr_ok( void *p, const char *file, int line )
{
    struct mem_debug *dl = mem_ptrtodl(p);
    if( dl->m.Mbeforeval != BEFOREVAL )
    {
	    if( file )
		fprintf( ferr, "%s: %d: ", file, line );
	    fprintf( ferr, "Pointer %p underrun\n", p );
	    mem_printdl(dl);
	    return 0;
    }
    if( memcmp(&dl->data[dl->m.Mnbytes],&afterval,sizeof(afterval)) )
    {
	    if( file )
		fprintf( ferr, "%s: %d: ", file, line );
	    fprintf( ferr, "Pointer %p overrun\n", p );
	    mem_printdl(dl);
	    return 0;
    }
    return 1;
}


static void mem_paranoid_check( const char *file, int line )
{
	register struct mem_debug *dl;

	for( dl = firstInList.Item.m.Mnext; dl; dl = dl->m.Mnext )
		if( !mem_allocated_ptr_ok( mem_dltoptr( dl ), file, line ) )
			_exit(1);
}


void mem_check()
{
	mem_paranoid_check( 0, 0 );
}

int mem_checkptr(void *p)
{
	register struct mem_debug *dl;

	for( dl = firstInList.Item.m.Mnext; dl; dl = dl->m.Mnext )
		if( p >= (void *)dl->data &&
		   p < (void *)( dl->data + dl->m.Mnbytes - 1 ) )
		    break;

	if( !dl ){
		fprintf( stderr, "%p: Invalid pointer\n", p );
	    return 0;
    }
	return mem_allocated_ptr_ok( mem_dltoptr( dl ), 0, 0 );
}

static void mem_init()
{
	if( !mem_inited )
	{
/*
		mem_count = 0;
		fp = NULL;
		mem_behavior = MEM_ABORT;
		mem_numalloc = 0;
		mem_maxalloc = 0;
		firstInList.Item.m.Mnext = NULL;
*/
		free(malloc(1));		/* initialize storage allocator	*/
	}
	mem_inited++;
}

/***************************/

static void mem_term()
{
	if( mem_inited == 1 )
	{
		register struct mem_debug *dl;
		int NullFiles = 0;

		for( dl = firstInList.Item.m.Mnext; dl; dl = dl->m.Mnext )
			if( dl->m.Mfile )
			{
				fputs( "Unfreed pointer: ", ferr );
				mem_printdl( dl );
			}
			else
				NullFiles++;

		if( NullFiles )
			fprintf( stderr, "%d remaining allocations from outside mem init\n",
					NullFiles );
	}
	mem_inited--;
	assert(mem_inited >= 0);
}

#else
int SomethingInMemObject = 0;
#endif /* MEM_DEBUG */
