//
//                      COPYRIGHT NOTICE
//
//      This code was originally written, and is owned by Brian Somers.
//      It may be copied, altered, given away or sold by anybody who
//      feels so inclined, but must at all times contain this copyright
//      notice.
//      brian@awfulhak.demon.co.uk
//
//
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "jobs.inc"
#	endif
#   endif
#else
#   include "jobs.h"
#   include "include/sys/wait.h"
#   include "include/unistd.h"
#   include "include/stdlib.h"
#   include "include/errno.h"
#   include "include/signal.h"
#endif

typedef job_proc *job_proc_ptr;

jobs::jobs(int parallel){
    int f;
    this->running = 0;
#ifdef MSDOS
    parallel = 1;
#else
    this->child = 0;
#endif
    this->max = parallel;
    if(this->max > 1){
	this->proc_list = mem_new job_proc_ptr[this->max];
	for(f = 0; f < this->max; f++) this->proc_list[f] = 0;
    }
}

jobs::~jobs(){
#ifndef MSDOS
    if( !this->child && this->max > 1){
	this->killall(SIGTERM);
	mem_delete this->proc_list;
    }
#endif
}

#ifdef MSDOS
int jobs::wait(action &){ return 0; }
#else
int jobs::wait(action &act){
    int status = 0;
    if(this->max > 1){
	pid_t pid;
	int f;
	if(!this->running) return 0;
	if(pid = ::wait(&status), pid == -1){
	    this->killall(SIGTERM);
	    perror( "wait" );
	    return JOB_FAILED_WAIT;
	}
	if( status )
	    return status;

	for(f = 0; f < this->max; f++)
	    if(this->proc_list[f] && this->proc_list[f]->pid == pid){
		if( !this->proc_list[f]->act.UnLock() && !status )
		    status = JOB_FAILED_UNLOCK;
		act = this->proc_list[f]->act;
		mem_delete this->proc_list[f];
		this->proc_list[f] = 0;
		this->running--;
		return status;
	    }
	fprintf( stderr, "Unexpected child process %ld finished\n", (long)pid );
    }
    return status;
}
#endif

int jobs::runable(int priority){
    for( int f = 0; f < this->max; f++ )
	if(this->proc_list[f] && this->proc_list[f]->priority < priority)
	    return 0;
    return 1;
}

#ifdef MSDOS
int jobs::execute(action &finished,action &act,const char *rmfmt,int,
		  int quiet,int doit){
#else
int jobs::execute(action &finished,action &act,const char *rmfmt,
		  int priority,int quiet,int doit){
#endif
    int ret;

#ifndef MSDOS
    if( this->max > 1 )
    {
		int num;
		pid_t pid;
		finished.reset();

		while( !this->runable( priority ) )
			if(ret = this->wait(finished), ret)
			{
				act.UnLock();
				return ret;
	    	}

		// Now, we only do it in the background if it's not priority zero !
		// Priority zero is special !
		if( priority )
		{
			this->running++;
			switch( pid = fork(), pid ){
				case -1:
					perror("fork");
					finished = act;
					return errno;
				case 0:
					this->child = 1;
					act.leave_lock();
					_exit( act.system( rmfmt, quiet, doit, " &" ) );
			}

			for(num = 0; num < this->max; num++)
		    	if( !this->proc_list[num] )
				{
					// Remember what I've spawned
					this->proc_list[num] =
						mem_new job_proc( pid, act, priority );
					break;
	    		}

			// If I've spawned max, Hang around for something to finish
			while( this->running == this->max )
				// Note, we'll never get in here and overwrite a valid
				// "finished" from above because running can't == max (one
				// finished, one started)!
				if(ret = this->wait(finished), ret)
				{
					act.UnLock();
					return ret;
	    		}
			return 0;
		}
    }
#endif
    ret = act.system( rmfmt, quiet, doit );
	if( !act.UnLock() && !ret )
		ret = JOB_FAILED_UNLOCK;
    finished = act;
    return ret;
}

#ifdef MSDOS
void jobs::killall(int){}
#else
void jobs::killall(int sig){
    int f;
    if(this->max > 1){
	for(f = 0; f < this->max && this->running; f++)
	    if(this->proc_list[f])
		if(!kill(this->proc_list[f]->pid,0))
		    kill(this->proc_list[f]->pid,sig);
	sleep(1);
	for(f = 0; f < this->max && this->running; f++)
	    if(this->proc_list[f])
		if(!kill(this->proc_list[f]->pid,0)){
		    kill(this->proc_list[f]->pid,SIGKILL);
		    sleep(1);
		    if(!kill(this->proc_list[f]->pid,0))
			fprintf( stderr, "Program %s pid %d won't die\n",
				this->proc_list[f]->act.prog(),
				(int)this->proc_list[f]->pid );
		    else{
			this->proc_list[f]->act.UnLock();
			this->proc_list[f] = 0;
			this->running--;
		    }
		}else{
		    this->proc_list[f]->act.UnLock();
		    this->proc_list[f] = 0;
		    this->running--;
		}
    }
}
#endif

int jobs::finish(action &finished){
    int ret;
    if(this->max > 1)
	while(this->running)
	    if(ret = this->wait(finished), ret) return ret;
    return 0;
}

#ifdef MSDOS
int jobs::isrunning(const action &) const { return 0; }
int jobs::waitfor(action &) { return 0; }
#else
int jobs::isrunning(const action &act) const {
    if(this->running)
    {
	int f;
	for(f = 0; f < this->max; f++)
	    if(this->proc_list[f]
		&& this->proc_list[f]->act.name() == act.name() )
	    break;
	if( f < this->max )
	    return f+1;		// see jobs::waitfor()
    }
    return 0;
}

int jobs::waitfor(action &act){
    int ret = 0;
    if(this->running)
    {
	int f = this->isrunning(act);
	if( f-- )
	    while( this->running && ( ret = this->wait( act ), !ret ) )
		if( !this->proc_list[f] )
		    break;
    }
    return ret;
}
#endif
