//
//                      COPYRIGHT NOTICE
//
//      This code was originally written, and is owned by Brian Somers.
//      It may be copied, altered, given away or sold by anybody who
//      feels so inclined, but must at all times contain this copyright
//      notice.
//      brian@awfulhak.demon.co.uk
//
//
#ifdef MK
#   if __GNUC__ < 2
#	if !defined(__ZTC__) && !defined(__WATCOMC__)
#	    include "module.inc"
#	endif
#   endif
#else
#   include "module.h"
#   include "pr.h"
#   include "include/stdio.h"
#   include "common/signals.h"
#   include "include/fcntl.h"
#   include "include/dos.h"
#   include "include/io.h"
#   include "include/prototyp.h"
#   include "include/dir.h"
#   include "include/memory.h"
#   include "include/unistd.h"
#endif

#if defined( MSDOS ) | defined( OS2 )
#	define SEP '\\'
#else
#	define SEP '/'
#endif

#define MOD_TOP_LEVEL (1)
#define MOD_CAN_BE_MAPPED (2)
#define MOD_VALID (4)
#define MOD_REVERSE_MAPPED (8)
#define MOD_HAS_MAPPED (16)
#define MOD_HAS_INTERFACE_MAPPED (32)
#define MOD_OBJECT_HALF_MAPPED (64)

#define VALID (this->flags & MOD_VALID)
#define MAPPED (this->mapped)
#define CAN_MAP (this->flags & MOD_CAN_BE_MAPPED && map)
#define TOP_LEVEL (this->flags & MOD_TOP_LEVEL)
#define REVERSE_MAPPED (this->flags & MOD_REVERSE_MAPPED)
#define HALF_MAPPED (this->half_mapped_name ? 1 : 0)
#define OBJECT_HALF_MAPPED (this->flags & MOD_OBJECT_HALF_MAPPED)
#define HAS_MAPPED (this->flags & MOD_HAS_MAPPED)
#define HAS_INTERFACE_MAPPED (this->flags & MOD_HAS_INTERFACE_MAPPED)

module::module(String &name,default_t &def,int top){
    this->compile = def.value(COMPILE);
    this->load = def.value(LOAD);
    this->includeline = def.value(INCLUDE_LINE);
    this->includelinecont = def.value(INCLUDE_LINE_CONT);
    this->responseline = def.value(RESPONSE_LINE);
    this->responselinecont = def.value(RESPONSE_LINE_CONT);
    this->object_namefmt = def.value(OBJECT_NAME);
    this->interface_namefmt = def.value(INTERFACE_NAME);
    this->code_namefmt = def.value(CODE_NAME);
    this->target = 0;
    this->derive = 0;
    this->flags = MOD_CAN_BE_MAPPED;
    if(top) this->flags |= MOD_TOP_LEVEL;
    this->has_mk = 0;
    this->mapped = this->half_mapped_name = 0;
    this->k.data = mem_malloc(this->k.length = name.len() + 1);
    memcpy(this->k.data,(char *)name,this->k.length);
    this->namebuf = &name;
}


module::~module(){
    if(this->derive) mem_delete this->derive;
    if(this->target) mem_delete this->target;
    if(this->mapped) mem_delete this->mapped;
    if(this->half_mapped_name) mem_delete this->half_mapped_name;
    mem_free( this->k.data );
}

int module::isvalid() const { return VALID; }
int module::ismapped() const { return this->mapped ? 1 : 0; }
int module::hasmapped() const { return HAS_MAPPED; }
int module::hasinterfacemapped() const { return HAS_INTERFACE_MAPPED; }
void module::now_hasmapped() { this->flags |= MOD_HAS_MAPPED; }
void module::now_hasinterfacemapped()
	{ this->flags |= MOD_HAS_INTERFACE_MAPPED; }
int module::objecthalfmapped() const { return OBJECT_HALF_MAPPED; }
void module::makevalid(){ this->flags |= MOD_VALID; this->namebuf = 0; }

static int MakeDirectory( String &Dir, int Silent ){
	struct stat st;
	int Result = 0;
	if( stat( Dir, &st ) == -1 ){
		int Sep = Dir.rchr( SEP );
		Dir.set( Sep, '\0' );
		Result = MakeDirectory( Dir, Silent );
		Dir.set( Sep, SEP );
		if( Result ){
			Dir.insert( 0, "mkdir " );
			if( !Silent )
				prcommand( Dir );
			system( Dir );
			Dir.del( 0, 6 );
			if( stat( Dir, &st ) == -1 || !(st.st_mode & S_IFDIR) )
				Result = 0;
		}
	}
	else if( st.st_mode & S_IFDIR )
		Result = 1;
	else
		fprintf( stderr, "%s: Not a directory\n", (char *)Dir );
	return Result;
}


int module::HalfMap(mapping *map, int Verbose, int Object) {
	if( Verbose ) fprintf( stderr, "Half map %s\n", this->name() );
	this->half_mapped_name = map->AnyMap( (char *)this->k.data );
	if(Object) this->flags |= MOD_OBJECT_HALF_MAPPED;	// even if halfmap fails
	if( HALF_MAPPED )
		this->flags &= ~MOD_CAN_BE_MAPPED;
	return HALF_MAPPED;
}

void module::reversemap(){
    if(!REVERSE_MAPPED) return;
    String *s = mem_new String((char *)this->k.data);
    mem_free( this->k.data );
    this->k.data = mem_malloc(this->k.length = this->mapped->len() + 1);
    memcpy( this->k.data,(char *)*this->mapped,this->k.length);
    mem_delete this->mapped;
    this->mapped = s;
    *this->namebuf = (const char *)this->k.data;
    this->flags &= ~MOD_REVERSE_MAPPED;
}

const char *module::realname() const {
    if(REVERSE_MAPPED) return (const char *)*this->mapped;
    return (const char *)k.data;
}

const char *module::name() const {
    if(this->mapped && !REVERSE_MAPPED) return (const char *)*this->mapped;
    return (const char *)this->k.data;
}

const char *module::object_prefix() const {
	return this->mapped && !REVERSE_MAPPED ? (const char *)*this->mapped
		: HALF_MAPPED ? (const char *)*this->half_mapped_name
			: (const char *)this->k.data;
}

// Note:  namefmt is only necessary for A_F calls that have no target->namefmt
time_t module::gettime( file_type type,mapping *map,int verbose,
						const char *namefmt){
    time_t *time;
    switch(type){
		case C_F:
	    time = &this->orig_code_time;
	    namefmt = this->code_namefmt;
	    break;

		case D_F:
	    time = &this->derive->src_time;
	    namefmt = this->derive->namefmt;
	    break;

		case H_F:
	    time = &this->orig_interface_time;
	    namefmt = this->interface_namefmt;
	    break;

		case O_F:
	    time = &this->orig_object_time;
	    namefmt = this->object_namefmt;
	    break;

		case A_F:
	    if(!this->target) return 0;
	    time = &this->target->orig_a_time;
	    if( this->target->namefmt ) namefmt = this->target->namefmt;
		else if( verbose && !namefmt )
			fputs( "Bad call to gettime: Missing namefmt\n", stderr );
	    break;

		default:
		return 0;
    }
	String *ModuleName = this->namebuf ?
		this->namebuf : mem_new String((const char *)this->k.data);
    String &name = this->mapped && !REVERSE_MAPPED ? *this->mapped :
		HALF_MAPPED && ( type == A_F || (type == O_F && OBJECT_HALF_MAPPED) ) ?
		*this->half_mapped_name : *ModuleName;
    int reversed = TOP_LEVEL;
    struct stat st;

    if(CAN_MAP && (this->mapped = map->Stat(name,namefmt,st,reversed), this->mapped)){
		if(verbose)
		{
			if(reversed)
				fprintf( stderr, "%s <= %s\n",
						(char *)*this->mapped, (char *)name );
			else
				fprintf( stderr, "%s => %s\n",
						(char *)name, (char *)*this->mapped );
			String FullName( *this->mapped );
			ExpandName( FullName, namefmt );
			fprintf( stderr, "gettime: stat %s\n", (char *)FullName ); 
		}
		this->flags &= ~MOD_CAN_BE_MAPPED;
		if(reversed) this->flags |= MOD_REVERSE_MAPPED;
		*time = st.st_mtime;
    }else{
		String FullName( name );
		ExpandName( FullName, namefmt );
		if( verbose )
			fprintf( stderr, "gettime: stat %s\n", (char *)FullName );
		if(stat(FullName,&st) == -1)
			*time = 0;
		else{
			*time = st.st_mtime;
			this->flags &= ~MOD_CAN_BE_MAPPED;
		}
    }

	if( this->mapped && type != O_F && type != A_F
		&& ( !this->derive || ( type != C_F && type != H_F ) ) )
	{
		String FullName( *ModuleName );
		ExpandName( FullName, namefmt );
		int Result = stat( FullName, &st);
		if( Result != -1 && st.st_mtime > *time )
			fprintf( stderr, "Warning: %s has been updated\n",
					(char *)FullName );
		else if( (Result == -1 && *time) || st.st_mtime + 20 < *time )
			this->flags |= (MOD_HAS_MAPPED|
				(type == H_F ? MOD_HAS_INTERFACE_MAPPED : 0));
	}
	if( !this->namebuf ) mem_delete ModuleName;
    return *time;
}

int module::openread( const String &namefmt, mapping *map, int verbose ){
    if(!this->namebuf) return -1;
    String &name = this->mapped && !REVERSE_MAPPED ? *this->mapped
		: *this->namebuf;
    int fd, reversed = TOP_LEVEL;

    if(CAN_MAP && (this->mapped = map->Open(name,namefmt,fd,reversed), this->mapped)){
		if(verbose)
			if( reversed )
				fprintf( stderr, "%s <= %s\n",
						(char *)*this->mapped, (char *)name );
			else
				fprintf( stderr, "%s => %s\n",
						(char *)name, (char *)*this->mapped );
		this->flags &= ~MOD_CAN_BE_MAPPED;
		if(reversed) this->flags |= MOD_REVERSE_MAPPED;
    }
	else
	{
		String FullName( name );
		ExpandName( FullName, namefmt );
		if(fd = open( FullName, O_RDONLY ), fd != -1)
			this->flags &= ~MOD_CAN_BE_MAPPED;
	}
    return fd;
}

int action::Setup( module *c, time_t t, char *namefmt, int Silent ){
    if( this->lock_namefmt ) this->UnLock();
    if( this->lockedby ) mem_free( this->lockedby );
    this->lockedby = 0;
    this->c = c;
    this->target_time = t;
    this->target_namefmt = namefmt;
    this->directory = this->c->HalfMappedName();
	if( !this->directory.len() || this->type == ACT_DERIVE ||
		( this->type == ACT_COMPILE && !this->c->objecthalfmapped() ) )
		this->directory = this->c->name();
    this->directory.setlen(this->directory.rchr( SEP ));
    this->exec_command.setlen(0);

	if( this->c->HalfMappedName() )
		return MakeDirectory( this->directory, Silent );
	return 1;
}

int action::Compiled( int Verbose ) const
{
    if( !this->target_namefmt )
	{
		if( Verbose )
			fprintf( stderr, "%s: No target namefmt: Not compiled !\n",
					this->c->name() );
		return 0;
	}
	file_type type;
	time_t *Real, Copy;

	if( this->type == ACT_LOAD )
	{
		type = A_F;
		Real = &this->c->target->orig_a_time;
	}
	else if( this->type == ACT_INTERNAL )
	{
		type = H_F;
		Real = &this->c->orig_interface_time;
	}
	else if( this->type == ACT_DERIVE )
	{
		type = H_F;
		Real = &this->c->orig_interface_time;
	}
	else
	{
		type = O_F;
	    Real = &this->c->orig_object_time;
	}
	Copy = *Real;
	int Result = this->target_time
		< this->c->gettime( type, 0, 0, this->target_namefmt );
	if( Verbose )
		fprintf( stderr, "%s: Recorded time: %ld, current time: %ld\n",
				this->c->name(), (long)Copy, (long)*Real );
	*Real = Copy;

	if( Result && this->type == ACT_DERIVE )
	{
		// Smelly, we've gotta do it again - just in case of a
		// crap derive instruction that only updates the C_F,
		// and not the H_F !
		Copy = this->c->orig_code_time;
		Result = this->target_time < this->c->gettime( C_F, 0, 0 );
		if( Verbose )
			fprintf( stderr, "%s: Recorded CODE time: %ld, current time: %ld\n",
					this->c->name(), (long)Copy,
					(long)this->c->orig_code_time );
		this->c->orig_code_time = Copy;
	}

	return Result;
}

const char *action::dir() const {
    if(!c) return 0;
    return (const char *)this->directory;
}

const char *action::command() const {
    return this->exec_command;
}

action &action::operator<<(const char *c){
    this->exec_command += c;
    return *this;
}

// Don't declare this inside ::prog - compilers such as WATCOM never
// delete it (stupid compiler).  This is probably because it doesn't actually
// instantiate it before main() and on exit doesn't really know what it did.
static String progReturn;

const char *action::prog() const {
    char ch;
    int f = -1;
    while(ch = this->exec_command[++f], ch && ch != ' ' && ch != '\t')
		;
    progReturn.setlen(0);
    progReturn.insert(0,this->exec_command,0,f);
    return progReturn;
}

void action::Remove(const char *namefmt,int silent) const {
    String target(this->c->HalfMappedName());
	if( !target.len() )
		target = this->name();
	ExpandName( target, namefmt );
    if( unlink( target ) ) perror( target );
    else if(!silent) fprintf( stderr, "Removed %s\n", (char *)target );
}

#ifndef MSDOS
static int Killed;
static Sig_Handler_Return CatchSig(int sig){
    kill(0,sig);
    Killed = sig;
}
#endif

// namefmt == file to remove
// quiet == shut up (don't say what I'm doing)
// doit == actually do it
// post == "End of output" string (not done when quiet)
int action::system
	(const char *namefmt,int quiet,int doit,const char *post) const
{
	switch( this->type )
	{
		case ACT_LOAD:
		case ACT_COMPILE:
		case ACT_DERIVE:
			break;

		default:
			return 0;
	}

    String command;
    size_t semi;
    int ret = 0;
    const char *start = this->exec_command;
#ifndef MSDOS
    signals sig(SIGHUP,SIGINT,SIGQUIT,SIGTERM,0);
#endif
	
    do
	{
		command = start;
		semi = 0;
		while(semi = command.chr( ';', semi ), semi != BAD_SIZE_T
			  && command[semi+1] == ';')
		{
			command.del(semi,semi+1);
			semi++;
			start++;				// so that it skips the ;; in the original
		}
		if(semi != BAD_SIZE_T)
		{
			command.setlen(semi);
			start += semi + 1;
		}
		command.unpad();
		
		if( !command.len() )
			continue;
		if(!quiet)
			pr2command( command, post ? post : "" );
#if !defined( MSDOS ) & !defined( OS2 )
		command.insert(0,"exec ");
		Killed = 0;
#ifdef __FreeBSD__
		setpgrp(0,getpid());
#else
		setpgrp();
#endif
		sig.save(CatchSig);
#endif
		if( doit &&
			( ( ret = ::system(command), ret & 255 ) || Killed ) )
		{
			if(this->Compiled())
				// Remove the target if a failed compilation updated it
				this->Remove( this->target_namefmt );

			if( namefmt )
				// Remove the lock file
				this->Remove( namefmt, 1 );
			sig.restore();
			return Killed ? Killed : ret;
		}
#ifndef MSDOS
		sig.restore();
#endif
	}while(semi != BAD_SIZE_T && !ret);

    if(namefmt)
		// Remove the lock file
		this->Remove( namefmt, 1 );

    if( doit && !this->Compiled() )
		ret = MODULE_NOT_COMPILED;
	
    return ret;
}

action &action::operator=(action &a){
    if( this->lock_namefmt ) this->UnLock();
    if( this->lockedby ) mem_free( this->lockedby );
	
    // Assign the lock to the lvalue
    this->lockedby = a.lockedby;
    this->lock_namefmt = a.lock_namefmt;
    a.lock_namefmt = a.lockedby = 0;
	
    this->type = a.type;
    this->exec_command = a.exec_command;
    this->directory = a.directory;
    this->c = a.c;
    this->target_namefmt = a.target_namefmt;
    this->target_time = a.target_time;
    return *this;
}

void action::reset(){
    if( this->lock_namefmt ) this->UnLock();
    this->c = 0;
    if( this->lockedby ) mem_free( this->lockedby );
    this->lockedby = 0;
    this->target_namefmt = 0;
    this->exec_command.setlen(0);
}

char *action::get_allocated_lockname(const char *namefmt) const {
    if( !namefmt || !this->c ) return 0;
	String NameFrom( this->c->object_prefix() );
	ExpandName( NameFrom, namefmt );
	char *Loc = (char *)mem_malloc( strlen( NameFrom ) + 1 );
	strcpy( Loc, NameFrom );
	return Loc;
}

int action::IsLocked( const char *namefmt )
{
    if( this->lock_namefmt )
		// Locked by me :)
		return 1;

    char *Name = this->get_allocated_lockname( namefmt );
	if( Name )
	{
		struct stat st;
		int File;
		if( this->lockedby )
			mem_free( this->lockedby );
		if( File = open( Name, O_RDONLY ), File == -1 ){
			this->lockedby = 0;
			mem_free( Name );
			return 0;
		}
		if( fstat( File, &st ) == -1 ){
			this->lockedby = (char *)mem_malloc( 21 );
			strcpy( this->lockedby, "an inaccessable file" );
		}else if( !st.st_size ){
			this->lockedby = (char *)mem_malloc( 14 );
			strcpy( this->lockedby, "an empty file" );
		}else if( st.st_size > 100 ){
			this->lockedby = (char *)mem_malloc( 44 );
			sprintf( this->lockedby, "a relatively large file (%ld bytes)\n",
					(long)st.st_size );
		}else{
			this->lockedby = (char *)mem_malloc( st.st_size + 1 );
			this->lockedby[read( File, this->lockedby, st.st_size )] = '\0';
		}
		close( File );
		mem_free( Name );
		return 1;
	}
	return 0;
}

int action::Lock( const char *namefmt ){
    if( this->lock_namefmt )
		return 1;
    int Result = 0;
    char *Name = this->get_allocated_lockname( namefmt );
    if( Name )
    {
	int LockFile;
	// this smelly code is for OS/2's broken NFS - you cant O_EXCL !
	// It'll usually work even if the file system implementation is crap
	if( !access( Name, 0 ) )
	    LockFile = -1;
	else
	    LockFile = sopen( Name, O_WRONLY|O_CREAT|O_EXCL, SH_DENYWR, 0666 );
	if( LockFile == -1 ){
	    if( !this->IsLocked( namefmt ) ){
		this->lockedby = (char *)mem_malloc( 21 );
		strcpy( this->lockedby, "an inaccessable file" );
	    }
	}else{
	    char *MyName = getenv("LOGNAME");
	    if( MyName )
		write( LockFile, MyName, strlen( MyName ) );
	    else
		write( LockFile, "<unknown>", 9 );
		this->lock_namefmt = (char *)mem_malloc( strlen( namefmt ) + 1 );
		strcpy( this->lock_namefmt, namefmt );
	    close( LockFile );
	    Result = 1;
	}
	mem_free( Name );
    }
    return Result;
}

int action::UnLock()
{
    int Result = 0;
    char *Name = this->get_allocated_lockname( this->lock_namefmt );
    if( Name )
    {
	if( Result = unlink( Name ) + 1, !Result )
	{
	    fprintf( stderr, "Failed to remove %s: ", Name );
	    perror( "" );
	}
	mem_free( Name );
	mem_free( this->lock_namefmt );
	this->lock_namefmt = 0;
    }
    return Result;
}

void action::leave_lock(){
    if( this->lock_namefmt ){
	mem_free( this->lock_namefmt );
	this->lock_namefmt = 0;
    }
}
